//--------------------------------------------------------------------------
//
//  Software for MSP430 based e-meters.
//
//  You may not use the Program in non-TI devices.
//
//  File: emeter-communications.c
//
//  Steve Underwood <steve-underwood@ti.com>
//  Texas Instruments Hong Kong Ltd.
//
//  $Id: emeter-communication.c,v 1.7 2005/11/17 09:25:23 a0754793 Exp $
//
/*! \file emeter-structs.h */
//
//--------------------------------------------------------------------------
//
#include <stdint.h>

#include "io.h"

#include <emeter-toolkit.h>
#include "emeter-structs.h"

#if !defined(NULL)
#define NULL    (void *) 0
#endif

#define FSEG_A 0x01080      // Flash Segment A start address
#define FSEG_B 0x01000      // Flash Segment B start address

#define TA_10MS 10460       //10ms timeout

#define ACK     0x55        // ЧӦ
#define NACK    0xaa        // ЧӦ
enum host_commands_e
{
//CALIBRATION REGISTER READ
    HOST_CMD_GET_METER_I1RMS_OFFSET             = 0x40,
    HOST_CMD_GET_METER_I2RMS_OFFSET             = 0x41,
    HOST_CMD_GET_METER_I1RMS_GAIN               = 0x42,
    HOST_CMD_GET_METER_I2RMS_GAIN               = 0x43,
    HOST_CMD_GET_METER_P1_GAIN                  = 0x44,
    HOST_CMD_GET_METER_P2_GAIN                  = 0x45,
    HOST_CMD_GET_METER_P1_PHASE                 = 0x46,
    HOST_CMD_GET_METER_P2_PHASE                 = 0x47,
    HOST_CMD_GET_METER_P1_OFFSET                = 0x48,
    HOST_CMD_GET_METER_P2_OFFSET                = 0x49,
    HOST_CMD_GET_METER_Q1_OFFSET                = 0x4A,
    HOST_CMD_GET_METER_Q2_OFFSET                = 0x4B,
    HOST_CMD_GET_METER_VRMS_GAIN                = 0x4C,
    //V5
    HOST_CMD_GET_READINGS_VDC                   = 0x4D,  
    HOST_CMD_GET_READINGS_I1DC                  = 0x4E,   
    HOST_CMD_GET_READINGS_I2DC                  = 0x4F,  

//METERING REGISTER READ
    HOST_CMD_GET_READINGS_P1                    = 0x60,
    HOST_CMD_GET_READINGS_P2                    = 0x61,
    HOST_CMD_GET_READINGS_Q1                    = 0x62,
    HOST_CMD_GET_READINGS_Q2                    = 0x63,
    HOST_CMD_GET_READINGS_S1                    = 0x64,
    HOST_CMD_GET_READINGS_S2                    = 0x65,
    HOST_CMD_GET_READINGS_VRMS                  = 0x66,
    HOST_CMD_GET_READINGS_FREG                  = 0x67,
    HOST_CMD_GET_READINGS_I1RMS                 = 0x68,
    HOST_CMD_GET_READINGS_I2RMS                 = 0x69,
    HOST_CMD_GET_READINGS_PF1                   = 0x6A,
    HOST_CMD_GET_READINGS_PF2                   = 0x6B,
    HOST_CMD_GET_READINGS_ACTENERGY1            = 0x6C,
    HOST_CMD_GET_READINGS_REACTENERGY1          = 0x6D,  
    //V5
    HOST_CMD_GET_READINGS_NEGTIVE_ACTENERGY1    = 0x6E,
    HOST_CMD_GET_READINGS_NEGTIVE_REACTENERGY1  = 0x6F,  
    HOST_CMD_GET_READINGS_VWFS                  = 0x70,  
    HOST_CMD_GET_READINGS_I1WFS                 = 0x71,  
    HOST_CMD_GET_READINGS_I2WFS                 = 0x72,      
//CONFIGURATION REGISTER READ   
    HOST_CMD_GET_METER_SYSCONF                  = 0x74,
    HOST_CMD_GET_METER_CSGCONF                  = 0x75,
    HOST_CMD_GET_METER_POWER_CONST              = 0x76,
    HOST_CMD_GET_METER_START_CURRENT            = 0x77,    
    HOST_CMD_GET_READINGS_IE                    = 0x78,   
    HOST_CMD_GET_READINGS_IFG                   = 0x79,   
    HOST_CMD_GET_READINGS_STATUS                = 0x7A,         
    HOST_CMD_GET_READINGS_CHECKSUM1             = 0x7B,
    HOST_CMD_GET_READINGS_WREN                  = 0x7C,
    HOST_CMD_GET_READINGS_METER_STATUS          = 0x7D,
    HOST_CMD_GET_READINGS_RXBUF                 = 0x7E,   
    HOST_CMD_GET_READINGS_TXBUF                 = 0x7F,         

   
//CALIBRATION REGISTER WRITE
    HOST_CMD_SET_METER_I1RMS_OFFSET             = 0x80,
    HOST_CMD_SET_METER_I2RMS_OFFSET             = 0x81,
    HOST_CMD_SET_METER_I1RMS_GAIN               = 0x82,
    HOST_CMD_SET_METER_I2RMS_GAIN               = 0x83,
    HOST_CMD_SET_METER_P1_GAIN                  = 0x84,
    HOST_CMD_SET_METER_P2_GAIN                  = 0x85,
    HOST_CMD_SET_METER_P1_PHASE                 = 0x86,
    HOST_CMD_SET_METER_P2_PHASE                 = 0x87,
    HOST_CMD_SET_METER_P1_OFFSET                = 0x88,
    HOST_CMD_SET_METER_P2_OFFSET                = 0x89,
    HOST_CMD_SET_METER_Q1_OFFSET                = 0x8A,
    HOST_CMD_SET_METER_Q2_OFFSET                = 0x8B,
    HOST_CMD_SET_METER_VRMS_GAIN                = 0x8C,  
    //V5
    HOST_CMD_SET_READINGS_VDC                   = 0x8D,  
    HOST_CMD_SET_READINGS_I1DC                  = 0x8E,   
    HOST_CMD_SET_READINGS_I2DC                  = 0x8F,      
//CONFIGURATION REGISTER WRITE    
    HOST_CMD_SET_METER_SYSCONF                  = 0xB4,
    HOST_CMD_SET_METER_CSGCONF                  = 0xB5,
    HOST_CMD_SET_METER_POWER_CONST              = 0xB6,
    HOST_CMD_SET_METER_START_CURRENT            = 0xB7,    
    HOST_CMD_SET_IE                             = 0xB8,       
    HOST_CMD_SET_WREN                           = 0xBC,   
    HOST_CMD_SET_SRST                           = 0xBD,          
//SPECIAL REGISTER WRITE 
};

#define CSG550_COMMAND_SIZE             1
#define MAX_CSG550_MSG_BODY_SIZE        6
#define TX_MASSAGE_LEN                  5
#define RX_MASSAGE_LEN                  4

typedef union
{
    unsigned char uint8[MAX_CSG550_MSG_BODY_SIZE];
    unsigned int uint16[MAX_CSG550_MSG_BODY_SIZE/2];
} serial_msg_t;

/* Incoming serial message buffer */
serial_msg_t rx_msg;
unsigned char rx_msg_ptr;

/* Outgoing serial message buffer */
serial_msg_t tx_msg;
static unsigned char tx_msg_ptr;

unsigned int *next_flash_loc;

static unsigned char host_command;
#ifndef  TXISRUSED
unsigned char receiving;
#endif


static void get_checksum()
{
  tx_msg.uint8[4] = host_command;
  tx_msg.uint8[4] += tx_msg.uint8[0];
  tx_msg.uint8[4] += tx_msg.uint8[1];
  tx_msg.uint8[4] += tx_msg.uint8[2];
  tx_msg.uint8[4] += tx_msg.uint8[3];
}

//////////////////////////Percy Yu 0902///////////////////////////////////////
//Function: Store the calibrated value to flash
//Parameter:  
//bi_work     1: 32bit calibration data
//            0: 16bit calibration data
//value       calibration data to be writen
//offset      Offset to 0x1000, with below format
//Data Format: Start from 0x1000          Offset(in 16bit int)
//    Live    Ac_offset                   0
//            I_rms_scale_factor          1
//            P_scale_factor              2
//            Phase_correction            3
//            Offset_active_power         4
//            Offset_reactive_power       5
//    Neutral Ac_offset                   0+6
//            I_rms_scale_factor          1+6
//            P_scale_factor              2+6
//            Phase_correction            3+6
//            Offset_active_power         4+6
//            Offset_reactive_power       5+6
//            V_rms_scale_factor          6+6
// Flash clock work at 400kHz. It took 36ms to finish this functions
//////////////////////////////////////////////////////////////////////////////
static void DataStore(unsigned char offset, int32_t value, unsigned char bi_word)
{
    int *read_ptr;
    int *write_ptr;
    int w;
    
    if((meter_status & WREN) == 0)//write enable dosent set, do not erease flash and return.
    {
        TXBUF0 = NACK;
        return;
    }
    else
    {
      TXBUF0 = ACK;
      meter_status &= ~WREN;    //disable write enable for next flash write
    }
    flash_clr((int *) FSEG_A);

    _DINT();
    //Set to write mode to prepare for copy
    FCTL3 = FWKEY;          /* Lock = 0 */
    FCTL1 = FWKEY | WRT;

    //Copy block B to A for back up
    read_ptr = (int *) FSEG_B;
    write_ptr = (int *) FSEG_A;
    for (w = 0;  w < 64;  w++)
        *write_ptr++ = *read_ptr++;
    
    flash_clr((int *) FSEG_B);

    //Set to write mode to prepare for copy
    FCTL3 = FWKEY;          /* Lock = 0 */
    FCTL1 = FWKEY | WRT;
    
    write_ptr = (int *) FSEG_B + offset;
    if(bi_word == 0)
      *write_ptr = (int)value&0xffff;
    else{
      *write_ptr++ = (int)value&0xffff;
      *write_ptr = (int)(value>>16);
    }
    
    //Copy block A to B, slipping in the new value at the right location
    read_ptr = (int *) FSEG_A;
    write_ptr = (int *) FSEG_B;
    for (w = 0;  w < 64;  w++, read_ptr++, write_ptr++)
    {
        if (*write_ptr == 0xffff)
            *write_ptr = *read_ptr;
    }
    flash_secure();
    _EINT();
    
    Checksum1_update();

}
//////////////////////////Percy Yu 1002///////////////////////////////////////
//Function: calibrate gain scale factor for I channel
//int error: the new P_scale_factor
//unsigned char ch: channel number  0,1(neutral)
//////////////////////////////////////////////////////////////////////////////
void Cal_Gain(short error, unsigned char ch){
/*  
    float tmp;      
    int value;
    struct current_sensor_nv_parms_s * phase_info;      
    unsigned char offset = P_SCALE_FACTOR_OFFSET;       
         
        if( ch == 1 )   //neutral 
        {
          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[1]; 
          offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W; 
        }
        else{
          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[0];  
        }
       
        value = phase_info->P_scale_factor[0];
        tmp = (value/(1+(signed short)error/(float)10000));
        value = (unsigned int)tmp;
        if(tmp - value > 0.5)
          value += 1;     
        
        DataStore(offset, value, 0);
*/
    unsigned char offset = P_SCALE_FACTOR_OFFSET;       
    #if defined(NEUTRAL_MONITOR_SUPPORT)   
    if( ch == 1 )   //neutral 
    {
      offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W;       
    }      
    #endif          
    DataStore(offset, error, 0);

}
 
//////////////////////////Percy Yu 1002///////////////////////////////////////
//Function: calibrate gain scale factor for Irms
//int error: the new Irms_scale_factor
//unsigned char ch: channel number  0,1(neutral)
//////////////////////////////////////////////////////////////////////////////
void Cal_IrmsGain(short error, unsigned char ch){

    unsigned char offset = I_RMS_SCALE_FACTOR_OFFSET;       
    #if defined(NEUTRAL_MONITOR_SUPPORT)   
    if( ch == 1 )   //neutral 
    {
      offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W;       
    }      
    #endif          
    DataStore(offset, error, 0);

}
//////////////////////////Percy Yu 1002///////////////////////////////////////
//Function: calibrate parameters in flash by word
//unsigned int error: the new value to be writen
//unsigned char offset: offset address from 0x1000, where we write new value
//////////////////////////////////////////////////////////////////////////////
void Cal_FlashParameter_Word(unsigned short error, unsigned char offset){
  
    DataStore(offset, error, 0);

}
void Cal_FlashParameter_Biword(unsigned long error, unsigned char offset){
        
    DataStore(offset, error, 1);

}
//////////////////////////Percy Yu 1002///////////////////////////////////////
//Function: calibrate phase shift for I channel
//char phase_error: the new phase shift from host, in terms of 1 step
//unsigned char ch: channel number  0,1(neutral)
//////////////////////////////////////////////////////////////////////////////
void Cal_Phase(short phase_error, unsigned char ch){
  
        struct current_sensor_nv_parms_s * phase_info;  
        unsigned char offset = PHASE_CORRECTION_OFFSET;  

        #if defined(NEUTRAL_MONITOR_SUPPORT) 
        if( ch == 1 )   //neutral 
        {
          offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W; 

          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[1]; 
//          value = phase_info->Phase_correction[0] - (signed short)phase_error;
          phase->current[1].in_phase_correction[0].step = I_HISTORY_STEPS - (phase_error>>8);
          phase->current[1].in_phase_correction[0].sd16_preloaded_offset = phase_error;
          SD24PRE_NEUTRAL = (phase_info->Phase_correction[0] - phase_error) & 0xFF;
        }
        else
        #endif
        {
          phase_info =(struct current_sensor_nv_parms_s *) &nv_parms.seg_a.s.chan1.current[0];  
//          value = phase_info->Phase_correction[0] - (signed short)phase_error;
          phase->current[0].in_phase_correction[0].step = I_HISTORY_STEPS - (phase_error>>8);
          phase->current[0].in_phase_correction[0].sd16_preloaded_offset = phase_error;
          SD24PRE_LIVE = (phase_info->Phase_correction[0] - phase_error) & 0xFF;
        }        
    
        DataStore(offset, phase_error, 0);          
}

//////////////////////////Percy Yu 1002///////////////////////////////////////
//Function: calibrate power offset for I channel
//int offset_error: the new poffset
//unsigned char ch: channel number  0,1(neutral)
//////////////////////////////////////////////////////////////////////////////
void Cal_Poffset(int offset_error, unsigned char ch){
        
    unsigned char offset = OFFSET_ACTIVE_POWER_OFFSET;   
           
    #if defined(NEUTRAL_MONITOR_SUPPORT)   
    if( ch == 1 )   //neutral 
    {
      offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W;       
    }      
    #endif         
    
    DataStore(offset, offset_error, 0);     
}

//////////////////////////Percy Yu 1002///////////////////////////////////////
//Function: calibrate reactive power offset for I channel
//int offset_error: the new poffset
//unsigned char ch: channel number  0,1(neutral)
//////////////////////////////////////////////////////////////////////////////
void Cal_Qoffset(int offset_error, unsigned char ch){
        
    unsigned char offset = OFFSET_REACTIVE_POWER_OFFSET;   
           
    #if defined(NEUTRAL_MONITOR_SUPPORT)   
    if( ch == 1 )   //neutral 
    {
      offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W;       
    }      
    #endif         
    
    DataStore(offset, offset_error, 0);     
}
//////////////////////////Percy Yu 1002///////////////////////////////////////
//Function: calibrate Irms offset for I channel
//int offset_error: the new poffset
//unsigned char ch: channel number  0,1(neutral)
//////////////////////////////////////////////////////////////////////////////
void Cal_Irmsoffset(int offset_error, unsigned char ch){
        
    unsigned char offset = AC_OFFSET_OFFSET;   
           
    #if defined(NEUTRAL_MONITOR_SUPPORT)   
    if( ch == 1 )   //neutral 
    {
      offset += CURRENT_SENSOR_NV_PARMS_S_SIZE_W;       
    }      
    #endif         
    
    DataStore(offset, offset_error, 0);     
}

//////////////////////////////////////////////////////////////////////////////
/////////////////////communication with host//////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void send_msg()
{
    tx_msg_ptr = 0;
    receiving = 0;   // رսմ
    // ֻһֽڣwhat......  Եģһֽһֽڷ...  906 918
    TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
    while((IFG1&UTXIFG0)==0);
    TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
    while((IFG1&UTXIFG0)==0);
    TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
    while((IFG1&UTXIFG0)==0);
    TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
    while((IFG1&UTXIFG0)==0);
    TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
    while((IFG1&UTXIFG0)==0);
    TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
    while((IFG1&UTXIFG0)==0);
}

void receive_msg()
{   
    receiving = 1;    
}
//This functions runs in RX interrupt service routine
char process_rx_message()
{
    char rt = 1;
    
    switch (host_command)
    {
///////////////send calibration data out//////////////
    case HOST_CMD_GET_METER_I1RMS_OFFSET:
      tx_msg.uint16[0] =  phase_nv->current[0].Ac_offset;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_I2RMS_OFFSET:
      tx_msg.uint16[0] =  phase_nv->current[1].Ac_offset;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_I1RMS_GAIN:
      tx_msg.uint16[0] =  phase_nv->current[0].I_rms_scale_factor[0];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_I2RMS_GAIN:
      tx_msg.uint16[0] =  phase_nv->current[1].I_rms_scale_factor[0];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_P1_GAIN:
      tx_msg.uint16[0] =  phase_nv->current[0].P_scale_factor[0];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_P2_GAIN:
      tx_msg.uint16[0] =  phase_nv->current[1].P_scale_factor[0];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_P1_PHASE:
      tx_msg.uint16[0] =  phase_nv->current[0].Phase_correction[0];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_P2_PHASE:
      tx_msg.uint16[0] =  phase_nv->current[1].Phase_correction[0];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_P1_OFFSET:
      tx_msg.uint16[0] =  phase_nv->current[0].Offset_active_power;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_P2_OFFSET:
      tx_msg.uint16[0] =  phase_nv->current[1].Offset_active_power;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_Q1_OFFSET:
      tx_msg.uint16[0] =  phase_nv->current[0].Offset_reactive_power;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_Q2_OFFSET:
      tx_msg.uint16[0] =  phase_nv->current[1].Offset_reactive_power;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_VRMS_GAIN:
      tx_msg.uint16[0] =  phase_nv->V_rms_scale_factor;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;         
//////////////////////////send metrology reading out/////////////////////
    case HOST_CMD_GET_READINGS_P1:
      tx_msg.uint16[0] = phase->active_power[0];
      tx_msg.uint16[1] = phase->active_power[0] >> 16;
      get_checksum();
      send_msg();
      break;          
    case HOST_CMD_GET_READINGS_P2:
      tx_msg.uint16[0] = phase->active_power[1];
      tx_msg.uint16[1] = phase->active_power[1] >> 16;
      get_checksum();
      send_msg();
      break;      
    case HOST_CMD_GET_READINGS_Q1:
      tx_msg.uint16[0] = phase->reactive_power[0];
      tx_msg.uint16[1] = phase->reactive_power[0] >> 16;
      get_checksum();
      send_msg();
      break;          
    case HOST_CMD_GET_READINGS_Q2:
      tx_msg.uint16[0] = phase->reactive_power[1];
      tx_msg.uint16[1] = phase->reactive_power[1] >> 16;
      get_checksum();
      send_msg();
      break;      
    case HOST_CMD_GET_READINGS_S1:
      tx_msg.uint16[0] = phase->apparent_power[0];
      tx_msg.uint16[1] = phase->apparent_power[0] >> 16;
      get_checksum();
      send_msg();
      break;          
    case HOST_CMD_GET_READINGS_S2:
      tx_msg.uint16[0] = phase->apparent_power[1];
      tx_msg.uint16[1] = phase->apparent_power[1] >> 16;
      get_checksum();
      send_msg();
      break;  
    case HOST_CMD_GET_READINGS_VRMS:
      tx_msg.uint16[0] = phase->V_rms;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;        
    case HOST_CMD_GET_READINGS_FREG:
      tx_msg.uint16[0] = phase->frequency;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break; 
    case HOST_CMD_GET_READINGS_I1RMS:
      tx_msg.uint16[0] = phase->I_rms[0];
      tx_msg.uint16[1] = phase->I_rms[0] >> 16;
      get_checksum();
      send_msg();
      break; 
    case HOST_CMD_GET_READINGS_I2RMS:
      tx_msg.uint16[0] = phase->I_rms[1];
      tx_msg.uint16[1] = phase->I_rms[1] >> 16;
      get_checksum();
      send_msg();
      break; 
    case HOST_CMD_GET_READINGS_PF1:
      tx_msg.uint16[0] = phase->power_factor[0];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break; 
    case HOST_CMD_GET_READINGS_PF2:
      tx_msg.uint16[0] = phase->power_factor[1];
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;                     
    case HOST_CMD_GET_READINGS_ACTENERGY1:
      tx_msg.uint16[0] = phase->energy.total_consumed_energy;
      tx_msg.uint16[1] = phase->energy.total_consumed_energy;
      send_msg();
      break;                
    case HOST_CMD_GET_READINGS_REACTENERGY1:
      tx_msg.uint16[0] = phase->energy.total_consumed_reactive_energy;
      tx_msg.uint16[1] = phase->energy.total_consumed_reactive_energy;
      get_checksum();
      send_msg();
      break;     
    case HOST_CMD_GET_READINGS_NEGTIVE_ACTENERGY1:
      tx_msg.uint16[0] = phase->energy.total_consumed_negtive_energy;
      tx_msg.uint16[1] = phase->energy.total_consumed_negtive_energy;
      send_msg();
      break;                
    case HOST_CMD_GET_READINGS_NEGTIVE_REACTENERGY1:
      tx_msg.uint16[0] = phase->energy.total_consumed_negtive_reactive_energy;
      tx_msg.uint16[1] = phase->energy.total_consumed_negtive_reactive_energy;
      get_checksum();
      send_msg();
      break;           
    case HOST_CMD_GET_READINGS_VDC:
      tx_msg.uint16[0] = phase->V_dc_estimate;
      tx_msg.uint16[1] = phase->V_dc_estimate >> 16;
      get_checksum();
      send_msg();
      break;    
    case HOST_CMD_GET_READINGS_I1DC:
      tx_msg.uint16[0] = phase->current[0].I_dc_estimate[0];
      tx_msg.uint16[1] = phase->current[0].I_dc_estimate[0] >> 16;
      get_checksum();
      send_msg();
      break; 
    case HOST_CMD_GET_READINGS_I2DC:
      tx_msg.uint16[0] = phase->current[1].I_dc_estimate[0];
      tx_msg.uint16[1] = phase->current[1].I_dc_estimate[0] >> 16;
      get_checksum();
      send_msg();
      break;       
    case  HOST_CMD_GET_READINGS_VWFS:
      tx_msg.uint16[0] = adc_buffer[0];
      tx_msg.uint16[1] = adc_buffer[0];
      get_checksum();
      send_msg();
      break;    
    case  HOST_CMD_GET_READINGS_I1WFS:
      tx_msg.uint16[0] = adc_buffer[1];
      tx_msg.uint16[1] = adc_buffer[0];
      get_checksum();
      send_msg();
      break; 
    case  HOST_CMD_GET_READINGS_I2WFS:
      tx_msg.uint16[0] = adc_buffer[2];
      tx_msg.uint16[1] = adc_buffer[0];
      get_checksum();
      send_msg();
      break;       
///////////////send configuration regisiter out//////////////      
    case HOST_CMD_GET_METER_SYSCONF:
      tx_msg.uint16[0] =  gSysconf;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_CSGCONF:
      tx_msg.uint16[0] =  gCsgconf;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_POWER_CONST:
      tx_msg.uint16[0] =  gPower_const;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;
    case HOST_CMD_GET_METER_START_CURRENT:
      tx_msg.uint16[0] =  gStart_curr;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;    
    case HOST_CMD_GET_READINGS_IE:
      tx_msg.uint16[0] = phase->int_enable;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;      
    case HOST_CMD_GET_READINGS_IFG:
      tx_msg.uint16[0] = phase->int_flag;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      phase->int_flag = 0;  
      meter_status &= ~INTPENDING;
//      end_interrupt();
      get_checksum();
      send_msg();
      break;     
      
    case HOST_CMD_GET_READINGS_STATUS:
      tx_msg.uint16[0] = phase->status;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;       

    case HOST_CMD_GET_READINGS_CHECKSUM1:
      tx_msg.uint16[0] = gChecksum1;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;      

    case HOST_CMD_GET_READINGS_WREN:
      tx_msg.uint16[0] = 0;
      if(meter_status&WREN) 
        tx_msg.uint16[0] |= 0x00CA;
      if(meter_status&WREN_REG)
        tx_msg.uint16[0] |= 0xC500;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break; 
      
    case HOST_CMD_GET_READINGS_METER_STATUS:
      tx_msg.uint16[0] = meter_status;
      tx_msg.uint16[1] = tx_msg.uint16[0];
      get_checksum();
      send_msg();
      break;        
      
    case HOST_CMD_GET_READINGS_RXBUF:
      tx_msg.uint16[0] = rx_msg.uint16[0];
      tx_msg.uint16[1] = rx_msg.uint16[1];
      get_checksum();
      send_msg();
      break;       

    case HOST_CMD_GET_READINGS_TXBUF:
      get_checksum();
      send_msg();
      break;    
      
////////////////////////set configuration////////////////////////////
    case HOST_CMD_SET_METER_SYSCONF:
      if(rx_msg.uint16[0] == rx_msg.uint16[1]){
        if(meter_status&WREN_REG){
          meter_status &= ~WREN_REG;
          gSysconf = rx_msg.uint16[0];
          switch_to_normal_mode();
        }
      }
      break;
    case HOST_CMD_SET_METER_CSGCONF:
      if(rx_msg.uint16[0] == rx_msg.uint16[1]){
        if(meter_status&WREN_REG){
          meter_status &= ~WREN_REG;
          gCsgconf = rx_msg.uint16[0];
          csg_config();
        }
      }
      break;
    case HOST_CMD_SET_METER_POWER_CONST:
      if(rx_msg.uint16[0] == rx_msg.uint16[1]){
        if(meter_status&WREN_REG){
          meter_status &= ~WREN_REG;
          gPower_const = rx_msg.uint16[0];
          csg_config();
        }
      }
      break;
    case HOST_CMD_SET_METER_START_CURRENT:
      if(rx_msg.uint16[0] == rx_msg.uint16[1])
        if(meter_status&WREN_REG){
          meter_status &= ~WREN_REG;
          gStart_curr = rx_msg.uint16[0];
        }
      break;        
    case HOST_CMD_SET_IE:
      if(rx_msg.uint16[0] == rx_msg.uint16[1])      
        if(meter_status&WREN_REG){
          meter_status &= ~WREN_REG;
          phase->int_enable = rx_msg.uint16[0];
        }
      break;       
    case HOST_CMD_SET_WREN:
      if(rx_msg.uint16[0] == rx_msg.uint16[1]){
        if(rx_msg.uint8[0]== 0xca)
          meter_status |= WREN;
        else
          meter_status &= ~WREN;
        if(rx_msg.uint8[1] == 0xc5)
          meter_status |= WREN_REG;
        else
          meter_status &= ~WREN_REG;
      }
      break;       
    case HOST_CMD_SET_SRST:
      if(rx_msg.uint16[0] == rx_msg.uint16[1]){
        if(rx_msg.uint16[0] == 0x0099){   
          _NOP();
          _NOP();
          WDTCTL = WDTCNTCL+WDTSSEL;        //  λ 1us
        }
      }
      break;       
       
/////////////////////////////////////set calibration data///////////////////////
    case HOST_CMD_SET_METER_I1RMS_OFFSET:
    case HOST_CMD_SET_METER_I2RMS_OFFSET:
      if(rx_msg.uint16[0] == rx_msg.uint16[1])
        Cal_Irmsoffset(rx_msg.uint16[0], host_command - HOST_CMD_SET_METER_I1RMS_OFFSET);
      else
        TXBUF0 = NACK;
      break;   
    case HOST_CMD_SET_METER_I1RMS_GAIN:
    case HOST_CMD_SET_METER_I2RMS_GAIN:
      if(rx_msg.uint16[0] == rx_msg.uint16[1] && rx_msg.uint16[0] < 0x8000)
        Cal_IrmsGain(rx_msg.uint16[0], host_command - HOST_CMD_SET_METER_I1RMS_GAIN);
      else
        TXBUF0 = NACK;
      break;   
    case HOST_CMD_SET_METER_P1_GAIN:     
    case HOST_CMD_SET_METER_P2_GAIN:
      if(rx_msg.uint16[0] == rx_msg.uint16[1] && rx_msg.uint16[0] < 0x8000)
        Cal_Gain(rx_msg.uint16[0], host_command - HOST_CMD_SET_METER_P1_GAIN);
      else
        TXBUF0 = NACK;
      break;
    case HOST_CMD_SET_METER_P1_PHASE:
    case HOST_CMD_SET_METER_P2_PHASE:
      if(rx_msg.uint16[0] == rx_msg.uint16[1])
        Cal_Phase(rx_msg.uint16[0], host_command - HOST_CMD_SET_METER_P1_PHASE);
      else
        TXBUF0 = NACK;
      break;      
    case HOST_CMD_SET_METER_P1_OFFSET:
    case HOST_CMD_SET_METER_P2_OFFSET:
      if(rx_msg.uint16[0] == rx_msg.uint16[1])
        Cal_Poffset(rx_msg.uint16[0], host_command - HOST_CMD_SET_METER_P1_OFFSET);
      else
        TXBUF0 = NACK;
      break;      
    case HOST_CMD_SET_METER_Q1_OFFSET:
    case HOST_CMD_SET_METER_Q2_OFFSET:
      if(rx_msg.uint16[0] == rx_msg.uint16[1])
        Cal_Qoffset(rx_msg.uint16[0], host_command - HOST_CMD_SET_METER_Q1_OFFSET);
      else
        TXBUF0 = NACK;
      break;   
    case HOST_CMD_SET_METER_VRMS_GAIN:
      if(rx_msg.uint16[0] == rx_msg.uint16[1] && rx_msg.uint16[0] < 0x8000)
        Cal_FlashParameter_Word(rx_msg.uint16[0], V_RMS_SCALE_FACTOR_OFFSET);
      else
        TXBUF0 = NACK;
      break;       
    //V5
    case HOST_CMD_SET_READINGS_VDC:
        Cal_FlashParameter_Biword(rx_msg.uint16[0], INIT_V_DC_OFFSET);
        break;   
    case HOST_CMD_SET_READINGS_I1DC:
        Cal_FlashParameter_Biword(rx_msg.uint16[0], INIT_I1_DC_OFFSET);
        break;   
    case HOST_CMD_SET_READINGS_I2DC:
        Cal_FlashParameter_Biword(rx_msg.uint16[0], INIT_I2_DC_OFFSET);
        break;   
      
    default:
      rt = 0;  
      TXBUF0 = NACK;
      break;
    }
    
    return rt;
}
////////////////////////////////////////////////////////////////////////
/////////////////////ISR////////////////////////////////////////////////
ISR(USART0RX, serial_rx_interrupt0)
{
    unsigned char ch, ch1;
//    unsigned char i;

//    DCOCTL = 0x6f;    //8MHz                      
//    BCSCTL1 = 0x8d;   //8MHz
//    for(i=0;i<255;i++);
    
    ch = RXBUF0;

//    DCOCTL = 0x7d;    //16MHz                      
//    BCSCTL1 = 0x8f;   //16MHz
    // ʱʱжʱֵSDжжchar_timeout_1107Լ
    /*
    char_timeout_1107 = SAMPLES_PER_10_SECONDS/200;     //time out is 40960/200/4096=0.05s  
    20110328*/
    char_timeout_1107 = 409;      // UART ʱж
    ch1 = ch&0xC0;
    
    if(receiving)
    {     
      rx_msg_ptr++;
      //        
      if(rx_msg_ptr <= 1){          // յ
        
        if(ch1==0x40){//read command,  process it
          //P1OUT ^= ~BIT3;//for test
          host_command = ch;
          rx_msg_ptr = 0;        // ռֵ
          process_rx_message();
        }
        // д
        else 
          if(ch1==0x80){//write command,  
            host_command = ch;           
        }
          // 룬־ʼ
          else{  //error command
            host_command = 0;           
            rx_msg_ptr = 0;    
          }
          
      }
      // ϴΪָΪдָν˷֧.
      else{//Or it's write command, we are reveicing the data to be written from host now
        rx_msg.uint8[rx_msg_ptr-2] = ch;      // յ
        if( rx_msg_ptr >  RX_MASSAGE_LEN ){   
          rx_msg_ptr = 0;
          process_rx_message();
        }
      }   
    }
    else{
#if 1      
      if((ch&0xC0) == 0x40){//read command
        host_command = ch;
        rx_msg_ptr = 0;
        process_rx_message();
      }
      else if((ch&0xC0) == 0x80){//new write command
        host_command = ch;
        rx_msg_ptr = 1;
        receive_msg(); 
      }
      else{

//        DCOCTL = 0x6f;    //8MHz                      
//        BCSCTL1 = 0x8d;   //8MHz
//        for(i=0;i<255;i++);
        
        TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
        if (tx_msg_ptr > TX_MASSAGE_LEN)
          receive_msg();   

//        DCOCTL = 0x7d;    //16MHz                      
//        BCSCTL1 = 0x8f;   //16MHz  
      }
#else
//        DCOCTL = 0x6f;    //8MHz                      
//        BCSCTL1 = 0x8d;   //8MHz
//        for(i=0;i<255;i++);

        TXBUF0 = tx_msg.uint8[tx_msg_ptr++];
        if (tx_msg_ptr > TX_MASSAGE_LEN)
          receive_msg();   

//        DCOCTL = 0x7d;    //16MHz  
//        BCSCTL1 = 0x8f;   //16MHz  
#endif
    } 
}